'use client';

import React, { createContext, useContext, ReactNode, useState, useCallback } from 'react';
import { siteConfig as initialSiteConfig } from '@/lib/data';
import { useDoc, useFirestore, useMemoFirebase } from '@/firebase';
import { doc } from 'firebase/firestore';
import { updateDocumentNonBlocking } from '@/firebase/non-blocking-updates';

interface SiteConfig {
  name: string;
  ceo: string;
  logo: string;
  location: string;
  established: string;
  phone: string;
  whatsappLink: string;
  googleMapsLink: string;
  googleMapsEmbed: string;
}

interface SiteConfigContextType {
  siteConfig: SiteConfig;
  setSiteConfig: (newConfig: Partial<SiteConfig>) => void;
  isLoading: boolean;
}

const SiteConfigContext = createContext<SiteConfigContextType | undefined>(undefined);

export const SiteConfigProvider = ({ children }: { children: ReactNode }) => {
  const [siteConfig, setConfig] = useState<SiteConfig>(initialSiteConfig);
  const [isLoading, setIsLoading] = useState(false);

  // This function can be used to update the local state.
  // To make it persistent, it would need to write back to a file or a database.
  // For now, it will just update the state for the current session.
  const setSiteConfig = useCallback((newConfig: Partial<SiteConfig>) => {
    setConfig(prevConfig => ({...prevConfig, ...newConfig}));
    
    // If we were still using firestore, we would do this:
    // const firestore = useFirestore();
    // const configDocRef = doc(firestore, 'siteConfig', 'default');
    // updateDocumentNonBlocking(configDocRef, newConfig);

  }, []);

  const value = {
    siteConfig,
    setSiteConfig,
    isLoading,
  };

  return (
    <SiteConfigContext.Provider value={value}>
      {children}
    </SiteConfigContext.Provider>
  );
};

export const useSiteConfig = () => {
  const context = useContext(SiteConfigContext);
  if (context === undefined) {
    throw new Error('useSiteConfig must be used within a SiteConfigProvider');
  }
  return context;
};
