'use client';

import { useMemo } from 'react';
import { useCollection, useFirestore, useMemoFirebase } from '@/firebase';
import { collection, query, orderBy } from 'firebase/firestore';
import { type Comment } from '@/lib/data';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Star, User } from 'lucide-react';
import { Avatar, AvatarFallback } from '@/components/ui/avatar';
import { formatDistanceToNow } from 'date-fns';
import { CommentForm } from './CommentForm';
import { Separator } from '../ui/separator';
import { Skeleton } from '../ui/skeleton';

interface CommentSectionProps {
  productId: string;
}

export function CommentSection({ productId }: CommentSectionProps) {
  const firestore = useFirestore();

  const commentsQuery = useMemoFirebase(() => {
    if (!firestore) return null;
    return query(
      collection(firestore, 'products', productId, 'comments'),
      orderBy('createdAt', 'desc')
    );
  }, [firestore, productId]);

  const { data: comments, isLoading } = useCollection<Comment>(commentsQuery);

  const averageRating = useMemo(() => {
    if (!comments || comments.length === 0) return 0;
    const total = comments.reduce((acc, comment) => acc + comment.rating, 0);
    return total / comments.length;
  }, [comments]);


  return (
    <div className="max-w-2xl mx-auto space-y-8">
      <h2 className="text-2xl font-bold tracking-tight">Customer Reviews</h2>
      
      {isLoading ? (
        <div>
            <Skeleton className="h-8 w-48 mb-4" />
            <div className="space-y-4">
                <Skeleton className="h-24 w-full" />
                <Skeleton className="h-24 w-full" />
            </div>
        </div>
      ) : (
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Star className="text-yellow-400 fill-yellow-400" />
              <span>
                {averageRating.toFixed(1)} out of 5
              </span>
            </CardTitle>
            <p className="text-sm text-muted-foreground">
              Based on {comments?.length ?? 0} reviews.
            </p>
          </CardHeader>
          <CardContent className="space-y-6">
            {comments && comments.length > 0 ? (
              comments.map((comment, index) => (
                <div key={comment.id}>
                    <div className="flex items-start gap-4">
                    <Avatar className="h-10 w-10 border">
                        <AvatarFallback>
                           <User />
                        </AvatarFallback>
                    </Avatar>
                    <div className="flex-1">
                        <div className="flex items-center justify-between">
                            <p className="font-semibold">{comment.authorName}</p>
                            <div className="flex items-center">
                                {[...Array(5)].map((_, i) => (
                                    <Star key={i} className={`h-4 w-4 ${i < comment.rating ? 'text-yellow-400 fill-yellow-400' : 'text-gray-300'}`} />
                                ))}
                            </div>
                        </div>
                        <p className="text-sm text-muted-foreground mt-0.5">
                        {comment.createdAt?.seconds ? formatDistanceToNow(new Date(comment.createdAt.seconds * 1000), { addSuffix: true }) : 'Just now'}
                        </p>
                        <p className="mt-2 text-foreground">{comment.text}</p>
                    </div>
                    </div>
                    {index < comments.length - 1 && <Separator className="mt-6" />}
                </div>
              ))
            ) : (
              <p className="text-muted-foreground text-center py-4">
                No reviews yet. Be the first to leave a review!
              </p>
            )}
          </CardContent>
        </Card>
      )}

      <Separator />

      <CommentForm productId={productId} />
    </div>
  );
}
