'use client';

import { useState } from 'react';
import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import { z } from 'zod';
import { collection, serverTimestamp } from 'firebase/firestore';
import { useFirestore, addDocumentNonBlocking } from '@/firebase';

import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from '@/components/ui/card';
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from '@/components/ui/form';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { Star, Loader2 } from 'lucide-react';
import { cn } from '@/lib/utils';
import { toast } from '@/hooks/use-toast';

const commentFormSchema = z.object({
  authorName: z.string().min(2, { message: 'Name must be at least 2 characters.' }),
  rating: z.number().min(1, { message: 'Please select a rating.' }).max(5),
  text: z.string().min(10, { message: 'Review must be at least 10 characters.' }),
});

type FormData = z.infer<typeof commentFormSchema>;

interface CommentFormProps {
  productId: string;
}

const EMOJIS = ['👍', '❤️', '🔥', '😄', '🎉'];

export function CommentForm({ productId }: CommentFormProps) {
  const firestore = useFirestore();
  const [hoveredRating, setHoveredRating] = useState(0);

  const form = useForm<FormData>({
    resolver: zodResolver(commentFormSchema),
    defaultValues: {
      authorName: '',
      rating: 0,
      text: '',
    },
  });

  const { isSubmitting } = form.formState;
  const currentRating = form.watch('rating');
  
  const handleEmojiClick = (emoji: string) => {
    const currentText = form.getValues('text');
    form.setValue('text', currentText + emoji, { shouldValidate: true });
  };

  const onSubmit = (data: FormData) => {
    if (!firestore) return;

    const commentsCollection = collection(firestore, 'products', productId, 'comments');
    const newComment = {
      ...data,
      createdAt: serverTimestamp(),
    };

    addDocumentNonBlocking(commentsCollection, newComment);
    
    toast({
      title: 'Review Submitted!',
      description: 'Thank you for your feedback.',
    });
    form.reset();
  };

  return (
    <Card>
      <Form {...form}>
        <form onSubmit={form.handleSubmit(onSubmit)}>
          <CardHeader>
            <CardTitle>Leave a Review</CardTitle>
            <CardDescription>Share your thoughts about this product with other customers.</CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <FormField
              control={form.control}
              name="authorName"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>Your Name</FormLabel>
                  <FormControl>
                    <Input placeholder="e.g., Jane Doe" {...field} />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />

            <FormField
              control={form.control}
              name="rating"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>Your Rating</FormLabel>
                  <FormControl>
                    <div
                      className="flex items-center gap-1"
                      onMouseLeave={() => setHoveredRating(0)}
                    >
                      {[1, 2, 3, 4, 5].map((star) => (
                        <Star
                          key={star}
                          className={cn(
                            'h-6 w-6 cursor-pointer transition-colors',
                            (hoveredRating >= star || currentRating >= star)
                              ? 'text-yellow-400 fill-yellow-400'
                              : 'text-gray-300'
                          )}
                          onMouseEnter={() => setHoveredRating(star)}
                          onClick={() => field.onChange(star)}
                        />
                      ))}
                    </div>
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />

            <FormField
              control={form.control}
              name="text"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>Your Review</FormLabel>
                  <FormControl>
                    <Textarea
                      placeholder="What did you like or dislike?"
                      className="min-h-[120px]"
                      {...field}
                    />
                  </FormControl>
                  <div className="flex items-center gap-2 pt-2">
                    <FormLabel className="text-xs text-muted-foreground">Add an emoji:</FormLabel>
                    {EMOJIS.map((emoji) => (
                      <button
                        key={emoji}
                        type="button"
                        onClick={() => handleEmojiClick(emoji)}
                        className="rounded-full p-2 text-xl transition-transform hover:scale-125 focus:outline-none focus:ring-2 focus:ring-ring"
                      >
                        {emoji}
                      </button>
                    ))}
                  </div>
                  <FormMessage />
                </FormItem>
              )}
            />
          </CardContent>
          <CardFooter>
            <Button type="submit" className="w-full" disabled={isSubmitting}>
              {isSubmitting && <Loader2 className="mr-2 h-4 w-4 animate-spin" />}
              Submit Review
            </Button>
          </CardFooter>
        </form>
      </Form>
    </Card>
  );
}
