'use client';

import { useState } from 'react';
import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import { z } from 'zod';
import { repairBookingFormSchema } from '@/lib/schemas';
import { useSiteConfig } from '@/hooks/use-site-config';

import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from '@/components/ui/card';
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from '@/components/ui/form';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { Progress } from '@/components/ui/progress';
import { Loader2 } from 'lucide-react';
import { cn } from '@/lib/utils';
import { toast } from '@/hooks/use-toast';

type FormData = z.infer<typeof repairBookingFormSchema>;

export function RepairBookingForm() {
  const { siteConfig } = useSiteConfig();
  const [step, setStep] = useState(1);
  
  const form = useForm<FormData>({
    resolver: zodResolver(repairBookingFormSchema),
    defaultValues: {
      name: '',
      phone: '',
      device: '',
      issue: '',
    },
  });

  const { isSubmitting } = form.formState;

  const onSubmit = (data: FormData) => {
    const message = `Hello, I'd like to book a repair.
*Name:* ${data.name}
*Phone:* ${data.phone}
*Device:* ${data.device}
*Issue:* ${data.issue}`;
    
    const whatsappUrl = `https://wa.me/${siteConfig.phone.replace('+', '')}?text=${encodeURIComponent(message)}`;
    window.open(whatsappUrl, '_blank');
    
    toast({
        title: 'Redirecting to WhatsApp',
        description: 'Please send the pre-filled message to confirm your repair request.',
    });
    
    form.reset();
    setStep(1);
  };
  
  const handleNext = async () => {
    let fieldsToValidate: ('name' | 'phone' | 'device' | 'issue')[] = [];
    if (step === 1) {
        fieldsToValidate = ['name', 'phone'];
    } else if (step === 2) {
        fieldsToValidate = ['device', 'issue'];
    }
  
    const isValid = await form.trigger(fieldsToValidate);
    if (isValid) {
      setStep(prev => prev + 1);
    }
  };

  const handleBack = () => {
    setStep(prev => prev - 1);
  };
  
  const progressValue = (step / 3) * 100;
  const formData = form.getValues();

  return (
    <Card>
      <CardHeader>
        <CardTitle>Book a Repair</CardTitle>
        <CardDescription>Follow the steps to submit your repair request.</CardDescription>
      </CardHeader>
      <Form {...form}>
        <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
          <CardContent className="space-y-6">
            <Progress value={progressValue} className="mb-6" />

            <div className={cn("space-y-4", step !== 1 && "hidden")}>
                <h3 className="font-semibold text-lg">Step 1: Your Contact Info</h3>
                <FormField
                  control={form.control}
                  name="name"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Full Name</FormLabel>
                      <FormControl>
                        <Input placeholder="e.g., John Doe" {...field} />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                <FormField
                  control={form.control}
                  name="phone"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Phone Number</FormLabel>
                      <FormControl>
                        <Input placeholder="e.g., 099123456" {...field} />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
            </div>
            
            <div className={cn("space-y-4", step !== 2 && "hidden")}>
                <h3 className="font-semibold text-lg">Step 2: Device & Issue</h3>
                <FormField
                  control={form.control}
                  name="device"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Device Model</FormLabel>
                      <FormControl>
                        <Input placeholder="e.g., iPhone 12 Pro" {...field} />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
                <FormField
                  control={form.control}
                  name="issue"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Issue Description</FormLabel>
                      <FormControl>
                        <Textarea placeholder="e.g., The screen is cracked and won't turn on." className="min-h-[100px]" {...field} />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
            </div>
            
            <div className={cn("space-y-4", step !== 3 && "hidden")}>
                <h3 className="font-semibold text-lg">Step 3: Review & Submit</h3>
                <div className="space-y-2 rounded-lg border bg-muted/50 p-4">
                    <div className="flex justify-between">
                        <span className="text-muted-foreground">Name:</span>
                        <span className="font-medium">{formData.name}</span>
                    </div>
                    <div className="flex justify-between">
                        <span className="text-muted-foreground">Phone:</span>
                        <span className="font-medium">{formData.phone}</span>
                    </div>
                    <div className="flex justify-between">
                        <span className="text-muted-foreground">Device:</span>
                        <span className="font-medium">{formData.device}</span>
                    </div>
                    <div className="flex justify-between items-start">
                        <span className="text-muted-foreground shrink-0 mr-4">Issue:</span>
                        <p className="font-medium text-right">{formData.issue}</p>
                    </div>
                </div>
                <p className="text-sm text-muted-foreground">
                    By clicking "Submit Request", you will be redirected to WhatsApp to send this information to us.
                </p>
            </div>

          </CardContent>
          <CardFooter className="flex justify-between">
              {step > 1 && (
                <Button type="button" variant="outline" onClick={handleBack}>
                  Back
                </Button>
              )}
              {step < 3 ? (
                <Button type="button" onClick={handleNext} className={step === 1 ? 'ml-auto' : ''}>
                  Next
                </Button>
              ) : (
                <Button type="submit" disabled={isSubmitting}>
                  {isSubmitting && <Loader2 className="mr-2 h-4 w-4 animate-spin" />}
                  Submit Request
                </Button>
              )}
          </CardFooter>
        </form>
      </Form>
    </Card>
  );
}
