'use client';

import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import { z } from 'zod';
import { useEffect, useState, useCallback } from 'react';
import { useDropzone } from 'react-dropzone';
import Image from 'next/image';

import { Button } from '@/components/ui/button';
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from '@/components/ui/form';
import { Input } from '@/components/ui/input';
import { Loader2, UploadCloud, X } from 'lucide-react';

const profileFormSchema = z.object({
  name: z.string().min(3, { message: 'Company name must be at least 3 characters.' }),
  ceo: z.string().min(3, { message: 'CEO name must be at least 3 characters.' }),
  logo: z.string().min(1, { message: 'Please upload a logo.' }),
});

type FormData = z.infer<typeof profileFormSchema>;

interface ProfileFormProps {
  isOpen: boolean;
  onClose: () => void;
  onSubmit: (data: FormData) => void;
  siteConfig: {
    name: string;
    ceo: string;
    logo: string;
  };
}

export function ProfileForm({ isOpen, onClose, onSubmit, siteConfig }: ProfileFormProps) {
  const form = useForm<FormData>({
    resolver: zodResolver(profileFormSchema),
    defaultValues: {
      name: '',
      ceo: '',
      logo: '',
    },
  });

  const [isUploading, setIsUploading] = useState(false);
  const [imagePreview, setImagePreview] = useState<string | null>(null);

  const resetFormState = useCallback(() => {
    if (siteConfig) {
      form.reset(siteConfig);
      setImagePreview(siteConfig.logo);
    }
    setIsUploading(false);
  }, [siteConfig, form]);

  useEffect(() => {
    if (isOpen) {
      resetFormState();
    }
  }, [siteConfig, isOpen, resetFormState]);


  const onDrop = useCallback((acceptedFiles: File[]) => {
    const file = acceptedFiles[0];
    if (!file) return;

    const reader = new FileReader();
    reader.onloadstart = () => {
      setIsUploading(true);
      setImagePreview(null);
    };
    reader.onloadend = (e) => {
      const dataUri = e.target?.result as string;
      if (dataUri) {
        setImagePreview(dataUri);
        form.setValue('logo', dataUri, { shouldValidate: true });
      }
      setIsUploading(false);
    };
    reader.readAsDataURL(file);
  }, [form]);

  const { getRootProps, getInputProps, isDragActive } = useDropzone({
    onDrop,
    accept: { 'image/*': ['.jpeg', '.png', '.gif', '.webp'] },
    multiple: false,
  });

  const handleClearImage = () => {
    form.setValue('logo', '', { shouldValidate: true });
    setImagePreview(null);
  };

  const handleFormSubmit = (data: FormData) => {
    onSubmit(data);
  };

  const isSubmitting = form.formState.isSubmitting;

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="sm:max-w-[525px]">
        <Form {...form}>
          <form onSubmit={form.handleSubmit(handleFormSubmit)} className="space-y-6">
            <DialogHeader>
              <DialogTitle>Edit Company Profile</DialogTitle>
              <DialogDescription>
                Update your company's information. This will be reflected across the site.
              </DialogDescription>
            </DialogHeader>

            <div className="grid gap-4 py-4">
            <FormField
                control={form.control}
                name="logo"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Company Logo</FormLabel>
                    <FormControl>
                      <div className="space-y-2">
                        {!imagePreview ? (
                          <div
                            {...getRootProps()}
                            className={`flex flex-col items-center justify-center p-10 border-2 border-dashed rounded-md cursor-pointer transition-colors ${
                              isDragActive ? 'border-primary bg-primary/10' : 'border-border hover:border-primary/50'
                            }`}
                          >
                            <input {...getInputProps()} />
                            {isUploading ? (
                              <div className="text-center">
                                <Loader2 className="mx-auto h-8 w-8 animate-spin" />
                                <p className="mt-2 text-sm text-muted-foreground">Uploading...</p>
                              </div>
                            ) : (
                              <div className="text-center">
                                <UploadCloud className="mx-auto h-8 w-8 text-muted-foreground" />
                                <p className="mt-2 text-sm text-muted-foreground">
                                  {isDragActive ? 'Drop the image here!' : "Drag 'n' drop an image, or click to select"}
                                </p>
                              </div>
                            )}
                          </div>
                        ) : (
                          <div className="relative w-24 h-24 mx-auto rounded-full overflow-hidden">
                            <Image src={imagePreview} alt="Logo preview" layout="fill" objectFit="cover" />
                            <Button
                              type="button"
                              variant="destructive"
                              size="icon"
                              className="absolute top-1 right-1 h-6 w-6"
                              onClick={handleClearImage}
                              disabled={isUploading}
                            >
                              <X className="h-4 w-4" />
                            </Button>
                          </div>
                        )}
                      </div>
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <FormField
                control={form.control}
                name="name"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Company Name</FormLabel>
                    <FormControl>
                      <Input placeholder="Your Company LLC" {...field} />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
              <FormField
                control={form.control}
                name="ceo"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>CEO Name</FormLabel>
                    <FormControl>
                      <Input placeholder="John Doe" {...field} />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </div>
            
            <DialogFooter>
              <Button type="button" variant="outline" onClick={onClose} disabled={isSubmitting || isUploading}>Cancel</Button>
              <Button type="submit" disabled={isSubmitting || isUploading || !form.formState.isValid}>
                {(isSubmitting || isUploading) && <Loader2 className="mr-2 h-4 w-4 animate-spin" />}
                Save Changes
              </Button>
            </DialogFooter>
          </form>
        </Form>
      </DialogContent>
    </Dialog>
  );
}
