'use client';

import { useState } from 'react';
import { useSiteConfig } from '@/hooks/use-site-config';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { ProfileForm } from './ProfileForm';
import Image from 'next/image';
import { Skeleton } from '../ui/skeleton';

export function ProfileEditor() {
  const { siteConfig, setSiteConfig, isLoading } = useSiteConfig();
  const [isFormOpen, setIsFormOpen] = useState(false);

  const handleFormSubmit = (newConfig: { name: string; ceo: string; logo: string; }) => {
    // This now updates the local state via the context provider.
    // To make this change permanent, the data would need to be saved
    // back to the local file, which is outside the scope of this interaction.
    setSiteConfig(newConfig);
    setIsFormOpen(false);
  };

  return (
    <Card>
      <CardHeader className="flex flex-row items-center justify-between">
        <CardTitle>Company Profile</CardTitle>
        <Button onClick={() => setIsFormOpen(true)}>Edit Profile</Button>
      </CardHeader>
      <CardContent className="space-y-6">
        {isLoading ? (
          <div className="flex items-center gap-6">
            <Skeleton className="h-20 w-20 rounded-full" />
            <div className="grid flex-1 grid-cols-2 gap-4">
                <div>
                    <Skeleton className="h-5 w-24 mb-2" />
                    <Skeleton className="h-7 w-48" />
                </div>
                <div>
                    <Skeleton className="h-5 w-16 mb-2" />
                    <Skeleton className="h-7 w-36" />
                </div>
            </div>
          </div>
        ) : (
          <div className="flex items-center gap-6">
            <Image src={siteConfig.logo} alt="Company Logo" width={80} height={80} className="rounded-full object-cover" />
            <div className="grid flex-1 grid-cols-2 gap-4">
              <div>
                <p className="text-sm font-medium text-muted-foreground">Company Name</p>
                <p className="text-lg font-semibold">{siteConfig.name}</p>
              </div>
              <div>
                <p className="text-sm font-medium text-muted-foreground">CEO</p>
                <p className="text-lg font-semibold">{siteConfig.ceo}</p>
              </div>
            </div>
          </div>
        )}
        <ProfileForm
          isOpen={isFormOpen}
          onClose={() => setIsFormOpen(false)}
          onSubmit={handleFormSubmit}
          siteConfig={siteConfig}
        />
      </CardContent>
    </Card>
  );
}
