'use client';

import { useState } from 'react';
import { useCollection, useFirestore, useMemoFirebase } from '@/firebase';
import { collection, doc, deleteDoc, updateDoc, addDoc } from 'firebase/firestore';
import { Product } from '@/lib/data';
import { generateReview, type GenerateReviewInput } from '@/ai/flows/generate-review-flow';


import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from '@/components/ui/table';
import { ProductForm } from './ProductForm';
import { Loader2, PlusCircle } from 'lucide-react';
import Image from 'next/image';
import { toast } from '@/hooks/use-toast';

export function ProductManager() {
  const firestore = useFirestore();
  const productsCollection = useMemoFirebase(
    () => collection(firestore, 'products'),
    [firestore]
  );
  const { data: products, isLoading: isLoadingProducts } = useCollection<Product>(productsCollection);

  const [isFormOpen, setIsFormOpen] = useState(false);
  const [editingProduct, setEditingProduct] = useState<Product | null>(null);

  const handleAdd = () => {
    setEditingProduct(null);
    setIsFormOpen(true);
  };

  const handleEdit = (product: Product) => {
    setEditingProduct(product);
    setIsFormOpen(true);
  };
  
  const handleDelete = async (productId: string) => {
    if (!firestore || !productId) return;
    try {
      await deleteDoc(doc(firestore, 'products', productId));
      toast({
        title: 'Product Deleted',
        description: 'The product has been successfully removed.',
      });
      setIsFormOpen(false); // Close the form on successful deletion
      setEditingProduct(null);
    } catch (error) {
      console.error('Error deleting product:', error);
      toast({
        variant: 'destructive',
        title: 'Error Deleting Product',
        description: 'Could not delete the product. Please try again.',
      });
    }
  };

  const handleFormSubmit = async (formData: Omit<Product, 'id'>) => {
    if (!firestore) return;

    if (editingProduct && editingProduct.id) {
      // Update existing product
      const productDoc = doc(firestore, 'products', editingProduct.id);
      try {
        await updateDoc(productDoc, formData);
        toast({
            title: 'Product Updated',
            description: 'The product has been successfully updated.',
        });
      } catch (error) {
        console.error('Error updating product:', error);
        toast({
            variant: 'destructive',
            title: 'Error Updating Product',
            description: 'Could not update the product. Please try again.',
        });
      }
    } else {
      // Add new product
      try {
        const reviewInput: GenerateReviewInput = {
          productName: formData.name,
          productDescription: formData.description,
        };
        const reviewResult = await generateReview(reviewInput);
        const productWithReview = {
          ...formData,
          review: reviewResult.review,
          reviewerName: reviewResult.reviewerName,
        };
        await addDoc(productsCollection, productWithReview);
        toast({
            title: 'Product Added',
            description: 'The new product has been successfully created.',
        });
      } catch (error) {
        console.error("Failed to generate review, adding product without it.", error);
        await addDoc(productsCollection, formData);
        toast({
            title: 'Product Added (without review)',
            description: 'The new product has been created, but the AI review failed.',
        });
      }
    }
    setIsFormOpen(false);
    setEditingProduct(null);
  };
  
  const isLoading = isLoadingProducts;

  return (
    <Card>
      <CardHeader className="flex flex-row items-center justify-between">
        <CardTitle>Products</CardTitle>
        <Button onClick={handleAdd}><PlusCircle className="mr-2" />Add Product</Button>
      </CardHeader>
      <CardContent>
        {isFormOpen && (
          <ProductForm
            isOpen={isFormOpen}
            onClose={() => setIsFormOpen(false)}
            onSubmit={handleFormSubmit}
            onDelete={handleDelete}
            product={editingProduct}
          />
        )}
        <div className="overflow-x-auto">
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>Image</TableHead>
                <TableHead>Name</TableHead>
                <TableHead>Category</TableHead>
                <TableHead className="w-[40%]">Description</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {isLoading ? (
                <TableRow>
                  <TableCell colSpan={4} className="text-center">
                    <Loader2 className="mx-auto h-8 w-8 animate-spin" />
                  </TableCell>
                </TableRow>
              ) : (
                products?.map((product) => (
                  <TableRow 
                    key={product.id}
                    onClick={() => handleEdit(product)}
                    className="cursor-pointer hover:bg-muted/50"
                  >
                    <TableCell>
                      <Image src={product.image} alt={product.name} width={60} height={60} className="rounded-md object-cover" />
                    </TableCell>
                    <TableCell className="font-medium">{product.name}</TableCell>
                    <TableCell>{product.category}</TableCell>
                    <TableCell className="text-sm text-muted-foreground truncate">{product.description}</TableCell>
                  </TableRow>
                ))
              )}
            </TableBody>
          </Table>
        </div>
      </CardContent>
    </Card>
  );
}
