'use client';

import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import { z } from 'zod';
import { useEffect, useState, useCallback } from 'react';
import { Product, categories } from '@/lib/data';
import { useDropzone } from 'react-dropzone';
import Image from 'next/image';

import { Button } from '@/components/ui/button';
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from '@/components/ui/form';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Loader2, UploadCloud, X, Trash2 } from 'lucide-react';
import { AlertDialog, AlertDialogAction, AlertDialogCancel, AlertDialogContent, AlertDialogDescription, AlertDialogFooter, AlertDialogHeader, AlertDialogTitle, AlertDialogTrigger } from '../ui/alert-dialog';


const productFormSchema = z.object({
  name: z.string().min(3, { message: 'Name must be at least 3 characters.' }),
  description: z.string().min(10, { message: 'Description must be at least 10 characters.' }),
  category: z.string().min(1, { message: 'Please select a category.' }),
  rating: z.coerce.number().min(1).max(5).optional(),
  image: z.string().min(1, { message: 'Please upload a main image.' }),
  image2: z.string().optional(),
  image3: z.string().optional(),
  image4: z.string().optional(),
});

type FormData = z.infer<typeof productFormSchema>;

interface ProductFormProps {
  isOpen: boolean;
  onClose: () => void;
  onSubmit: (data: Omit<Product, 'id'>) => void;
  onDelete: (productId: string) => void;
  product?: Product | null;
}

const ImageUploader = ({ form, fieldName, label }: { form: any, fieldName: "image" | "image2" | "image3" | "image4", label: string }) => {
    const [isUploading, setIsUploading] = useState(false);
    const [imagePreview, setImagePreview] = useState<string | null>(form.getValues(fieldName));
  
    useEffect(() => {
      setImagePreview(form.getValues(fieldName));
    }, [form, fieldName]);
  
    const onDrop = useCallback((acceptedFiles: File[]) => {
      const file = acceptedFiles[0];
      if (!file) return;
  
      const reader = new FileReader();
      reader.onloadstart = () => {
        setIsUploading(true);
        setImagePreview(null);
      };
      reader.onloadend = (e) => {
        const dataUri = e.target?.result as string;
        if (dataUri) {
          setImagePreview(dataUri);
          form.setValue(fieldName, dataUri, { shouldValidate: true });
        }
        setIsUploading(false);
      };
      reader.readAsDataURL(file);
    }, [form, fieldName]);
  
    const { getRootProps, getInputProps, isDragActive } = useDropzone({
      onDrop,
      accept: { 'image/*': ['.jpeg', '.png', '.gif', '.webp'] },
      multiple: false,
    });
  
    const handleClearImage = () => {
      form.setValue(fieldName, '', { shouldValidate: true });
      setImagePreview(null);
    };
  
    return (
      <FormItem>
        <FormLabel>{label}</FormLabel>
        <FormControl>
          <div className="space-y-2">
            {!imagePreview ? (
              <div
                {...getRootProps()}
                className={`flex flex-col items-center justify-center p-10 border-2 border-dashed rounded-md cursor-pointer transition-colors ${
                  isDragActive ? 'border-primary bg-primary/10' : 'border-border hover:border-primary/50'
                }`}
              >
                <input {...getInputProps()} />
                {isUploading ? (
                  <div className="text-center">
                    <Loader2 className="mx-auto h-8 w-8 animate-spin" />
                    <p className="mt-2 text-sm text-muted-foreground">Uploading...</p>
                  </div>
                ) : (
                  <div className="text-center">
                    <UploadCloud className="mx-auto h-8 w-8 text-muted-foreground" />
                    <p className="mt-2 text-sm text-muted-foreground">
                      {isDragActive ? 'Drop the image here!' : "Drag 'n' drop an image, or click to select"}
                    </p>
                  </div>
                )}
              </div>
            ) : (
              <div className="relative w-full aspect-video rounded-md overflow-hidden">
                <Image src={imagePreview} alt="Product preview" layout="fill" objectFit="cover" />
                <Button
                  type="button"
                  variant="destructive"
                  size="icon"
                  className="absolute top-2 right-2 h-7 w-7"
                  onClick={handleClearImage}
                  disabled={isUploading}
                >
                  <X className="h-4 w-4" />
                </Button>
              </div>
            )}
          </div>
        </FormControl>
        <FormMessage />
      </FormItem>
    );
  };


export function ProductForm({ isOpen, onClose, onSubmit, onDelete, product }: ProductFormProps) {
  const form = useForm<FormData>({
    resolver: zodResolver(productFormSchema),
    defaultValues: {
      name: '',
      description: '',
      category: '',
      rating: 5,
      image: '',
      image2: '',
      image3: '',
      image4: '',
    },
  });

  const resetFormState = useCallback(() => {
    if (product) {
      form.reset({
        ...product,
        rating: product.rating || 5,
      });
    } else {
      form.reset({ name: '', description: '', category: '', rating: 5, image: '', image2: '', image3: '', image4: '' });
    }
  }, [product, form]);

  useEffect(() => {
    if (isOpen) {
      resetFormState();
    }
  }, [product, isOpen, resetFormState]);


  const handleFormSubmit = (data: FormData) => {
    onSubmit(data);
  };

  const isSubmitting = form.formState.isSubmitting;

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="sm:max-w-[525px]">
        <Form {...form}>
          <form onSubmit={form.handleSubmit(handleFormSubmit)} className="space-y-6">
            <DialogHeader>
              <DialogTitle>{product ? 'Edit Product' : 'Add New Product'}</DialogTitle>
              <DialogDescription>
                Fill in the details below to {product ? 'update the' : 'add a new'} product.
              </DialogDescription>
            </DialogHeader>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-4 py-4 max-h-[70vh] overflow-y-auto pr-2">
               <div className="md:col-span-2">
                 <FormField
                    control={form.control}
                    name="name"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Product Name</FormLabel>
                        <FormControl>
                          <Input placeholder="e.g., iPhone 13 Pro Screen" {...field} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
               </div>
              
               <div className="md:col-span-1">
                <FormField
                    control={form.control}
                    name="category"
                    render={({ field }) => (
                    <FormItem>
                        <FormLabel>Category</FormLabel>
                        <Select onValueChange={field.onChange} defaultValue={field.value}>
                        <FormControl>
                            <SelectTrigger>
                            <SelectValue placeholder="Select a category" />
                            </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                            {categories.map((cat) => (
                            <SelectItem key={cat.id} value={cat.id}>
                                {cat.name}
                            </SelectItem>
                            ))}
                        </SelectContent>
                        </Select>
                        <FormMessage />
                    </FormItem>
                    )}
                />
               </div>

                <div className="md:col-span-1">
                    <FormField
                        control={form.control}
                        name="rating"
                        render={({ field }) => (
                        <FormItem>
                            <FormLabel>Star Rating</FormLabel>
                            <Select onValueChange={field.onChange} defaultValue={String(field.value)}>
                            <FormControl>
                                <SelectTrigger>
                                <SelectValue placeholder="Select a rating" />
                                </SelectTrigger>
                            </FormControl>
                            <SelectContent>
                                {[5, 4, 3, 2, 1].map(star => (
                                <SelectItem key={star} value={String(star)}>
                                    {star} Star{star > 1 && 's'}
                                </SelectItem>
                                ))}
                            </SelectContent>
                            </Select>
                            <FormMessage />
                        </FormItem>
                        )}
                    />
                </div>
              
               <div className="md:col-span-2">
                <FormField
                    control={form.control}
                    name="description"
                    render={({ field }) => (
                    <FormItem>
                        <FormLabel>Product Description</FormLabel>
                        <FormControl>
                        <Textarea placeholder="A short description of the product..." {...field} />
                        </FormControl>
                        <FormMessage />
                    </FormItem>
                    )}
                />
               </div>
               <ImageUploader form={form} fieldName="image" label="Main Image" />
               <ImageUploader form={form} fieldName="image2" label="Image 2" />
               <ImageUploader form={form} fieldName="image3" label="Image 3" />
               <ImageUploader form={form} fieldName="image4" label="Image 4" />
            </div>
            
            <DialogFooter className="justify-between">
              <div>
                {product && (
                     <AlertDialog>
                        <AlertDialogTrigger asChild>
                           <Button type="button" variant="destructive">
                             <Trash2 className="mr-2 h-4 w-4" />
                              Delete
                           </Button>
                        </AlertDialogTrigger>
                        <AlertDialogContent>
                            <AlertDialogHeader>
                                <AlertDialogTitle>Are you absolutely sure?</AlertDialogTitle>
                                <AlertDialogDescription>
                                This action cannot be undone. This will permanently delete the product.
                                </AlertDialogDescription>
                            </AlertDialogHeader>
                            <AlertDialogFooter>
                                <AlertDialogCancel>Cancel</AlertDialogCancel>
                                <AlertDialogAction onClick={() => onDelete(product.id!)}>Continue</AlertDialogAction>
                            </AlertDialogFooter>
                        </AlertDialogContent>
                      </AlertDialog>
                )}
              </div>
              <div className="flex gap-2">
                <Button type="button" variant="outline" onClick={onClose} disabled={isSubmitting}>Cancel</Button>
                <Button type="submit" disabled={isSubmitting || !form.formState.isValid}>
                    {isSubmitting && <Loader2 className="mr-2 h-4 w-4 animate-spin" />}
                    Save Product
                </Button>
              </div>
            </DialogFooter>
          </form>
        </Form>
      </DialogContent>
    </Dialog>
  );
}
