'use client';

import { useState } from 'react';
import { useCollection, useFirestore, useMemoFirebase } from '@/firebase';
import { addDocumentNonBlocking, deleteDocumentNonBlocking, updateDocumentNonBlocking } from '@/firebase/non-blocking-updates';
import { collection, doc } from 'firebase/firestore';
import { GalleryImage } from '@/lib/data';

import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from '@/components/ui/table';
import { GalleryForm } from './GalleryForm';
import { AlertDialog, AlertDialogAction, AlertDialogCancel, AlertDialogContent, AlertDialogDescription, AlertDialogFooter, AlertDialogHeader, AlertDialogTitle, AlertDialogTrigger } from '../ui/alert-dialog';
import { Loader2, Pencil, PlusCircle, Trash2 } from 'lucide-react';
import Image from 'next/image';

export function GalleryManager() {
  const firestore = useFirestore();
  const galleryCollection = useMemoFirebase(
    () => collection(firestore, 'gallery'),
    [firestore]
  );
  const { data: images, isLoading } = useCollection<GalleryImage>(galleryCollection);

  const [isFormOpen, setIsFormOpen] = useState(false);
  const [editingImage, setEditingImage] = useState<GalleryImage | null>(null);

  const handleAdd = () => {
    setEditingImage(null);
    setIsFormOpen(true);
  };

  const handleEdit = (image: GalleryImage) => {
    setEditingImage(image);
    setIsFormOpen(true);
  };
  
  const handleDelete = (imageId: string) => {
    if (!imageId) return;
    const imageDoc = doc(firestore, 'gallery', imageId);
    deleteDocumentNonBlocking(imageDoc);
  };

  const handleFormSubmit = (formData: Omit<GalleryImage, 'id'>) => {
    if (editingImage && editingImage.id) {
      // Update existing image
      const imageDoc = doc(firestore, 'gallery', editingImage.id);
      updateDocumentNonBlocking(imageDoc, formData);
    } else {
      // Add new image
      addDocumentNonBlocking(galleryCollection, formData);
    }
    setIsFormOpen(false);
    setEditingImage(null);
  };

  return (
    <Card>
      <CardHeader className="flex flex-row items-center justify-between">
        <CardTitle>Gallery</CardTitle>
        <Button onClick={handleAdd}><PlusCircle className="mr-2" />Add Image</Button>
      </CardHeader>
      <CardContent>
        <GalleryForm
          isOpen={isFormOpen}
          onClose={() => setIsFormOpen(false)}
          onSubmit={handleFormSubmit}
          image={editingImage}
        />
        <div className="overflow-x-auto">
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>Image</TableHead>
                <TableHead>Name</TableHead>
                <TableHead>Role</TableHead>
                <TableHead>Description</TableHead>
                <TableHead className="text-right">Actions</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {isLoading ? (
                <TableRow>
                  <TableCell colSpan={5} className="text-center">
                    <Loader2 className="mx-auto h-8 w-8 animate-spin" />
                  </TableCell>
                </TableRow>
              ) : (
                images?.map((image) => (
                  <TableRow key={image.id}>
                    <TableCell>
                        <Image src={image.imageUrl} alt={image.description} width={80} height={80} className="rounded-md object-cover" />
                    </TableCell>
                    <TableCell className="font-medium">{image.name}</TableCell>
                    <TableCell>{image.role}</TableCell>
                    <TableCell className="text-sm text-muted-foreground">{image.description}</TableCell>
                    <TableCell className="text-right">
                      <Button variant="ghost" size="icon" onClick={() => handleEdit(image)}>
                        <Pencil />
                      </Button>
                      <AlertDialog>
                        <AlertDialogTrigger asChild>
                           <Button variant="ghost" size="icon" className="text-destructive hover:text-destructive">
                             <Trash2 />
                           </Button>
                        </AlertDialogTrigger>
                        <AlertDialogContent>
                            <AlertDialogHeader>
                                <AlertDialogTitle>Are you absolutely sure?</AlertDialogTitle>
                                <AlertDialogDescription>
                                This action cannot be undone. This will permanently delete the image from the gallery.
                                </AlertDialogDescription>
                            </AlertDialogHeader>
                            <AlertDialogFooter>
                                <AlertDialogCancel>Cancel</AlertDialogCancel>
                                <AlertDialogAction onClick={() => handleDelete(image.id!)}>Continue</AlertDialogAction>
                            </AlertDialogFooter>
                        </AlertDialogContent>
                      </AlertDialog>
                    </TableCell>
                  </TableRow>
                ))
              )}
            </TableBody>
          </Table>
        </div>
      </CardContent>
    </Card>
  );
}
