'use client';

import { useForm, Controller } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import { z } from 'zod';
import { useEffect, useState, useCallback, useMemo } from 'react';
import { GalleryImage } from '@/lib/data';
import { analyzeImage, AnalyzeImageOutput } from '@/ai/flows/analyze-image-flow';
import { useDropzone } from 'react-dropzone';

import { Button } from '@/components/ui/button';
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from '@/components/ui/form';
import { Input } from '@/components/ui/input';
import { Loader2, UploadCloud, X } from 'lucide-react';
import Image from 'next/image';
import { Textarea } from '../ui/textarea';

const galleryFormSchema = z.object({
  imageUrl: z.string().min(1, { message: 'Please upload an image.' }),
  description: z.string().min(3, { message: 'Description must be at least 3 characters.' }),
  imageHint: z.string().min(2, { message: 'Hint must be at least 2 characters.' }),
  name: z.string().optional(),
  role: z.string().optional(),
});

type FormData = z.infer<typeof galleryFormSchema>;

interface GalleryFormProps {
  isOpen: boolean;
  onClose: () => void;
  onSubmit: (data: Omit<GalleryImage, 'id'>) => void;
  image?: GalleryImage | null;
}

export function GalleryForm({ isOpen, onClose, onSubmit, image }: GalleryFormProps) {
  const form = useForm<FormData>({
    resolver: zodResolver(galleryFormSchema),
    defaultValues: {
      imageUrl: '',
      description: '',
      imageHint: '',
      name: '',
      role: '',
    },
  });

  const [isAnalyzing, setIsAnalyzing] = useState(false);
  const [imagePreview, setImagePreview] = useState<string | null>(null);

  const resetFormState = useCallback(() => {
    if (image) {
      form.reset(image);
      setImagePreview(image.imageUrl);
    } else {
      form.reset({ imageUrl: '', description: '', imageHint: '', name: '', role: '' });
      setImagePreview(null);
    }
    setIsAnalyzing(false);
  }, [image, form]);

  useEffect(() => {
    if (isOpen) {
      resetFormState();
    }
  }, [image, isOpen, resetFormState]);


  const onDrop = useCallback(async (acceptedFiles: File[]) => {
    const file = acceptedFiles[0];
    if (!file) return;

    const reader = new FileReader();
    reader.onloadstart = () => {
      setIsAnalyzing(true);
      setImagePreview(null);
    };
    reader.onloadend = async (e) => {
      const dataUri = e.target?.result as string;
      if (dataUri) {
        setImagePreview(dataUri);
        try {
          const analysis: AnalyzeImageOutput = await analyzeImage({ photoDataUri: dataUri });
          form.setValue('imageUrl', dataUri);
          form.setValue('description', analysis.description, { shouldValidate: true });
          form.setValue('imageHint', analysis.imageHint, { shouldValidate: true });
        } catch (error) {
          console.error("AI analysis failed:", error);
          form.setValue('imageUrl', dataUri); // Still set image on failure
        } finally {
          setIsAnalyzing(false);
        }
      }
    };
    reader.readAsDataURL(file);
  }, [form]);

  const { getRootProps, getInputProps, isDragActive } = useDropzone({
    onDrop,
    accept: { 'image/*': ['.jpeg', '.png', '.gif', '.webp'] },
    multiple: false,
  });

  const handleClearImage = () => {
    form.reset({ imageUrl: '', description: '', imageHint: '', name: '', role: ''});
    setImagePreview(null);
  }

  const handleFormSubmit = (data: FormData) => {
    onSubmit(data);
  };

  const isSubmitting = form.formState.isSubmitting;

  return (
    <Dialog open={isOpen} onOpenChange={(open) => !open && onClose()}>
      <DialogContent className="sm:max-w-[525px]">
        <Form {...form}>
          <form onSubmit={form.handleSubmit(handleFormSubmit)} className="space-y-6">
            <DialogHeader>
              <DialogTitle>{image ? 'Edit Image' : 'Add New Image'}</DialogTitle>
              <DialogDescription>
                {image ? 'Update the details below.' : 'Upload an image and the AI will help describe it.'}
              </DialogDescription>
            </DialogHeader>

            <div className="grid gap-4 py-4 max-h-[70vh] overflow-y-auto pr-2">
              <FormField
                control={form.control}
                name="imageUrl"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Image Upload</FormLabel>
                    <FormControl>
                        <div className="space-y-2">
                           {!imagePreview ? (
                             <div
                                {...getRootProps()}
                                className={`flex flex-col items-center justify-center p-10 border-2 border-dashed rounded-md cursor-pointer transition-colors ${
                                isDragActive ? 'border-primary bg-primary/10' : 'border-border hover:border-primary/50'
                                }`}
                            >
                                <input {...getInputProps()} />
                                {isAnalyzing ? (
                                    <div className="text-center">
                                        <Loader2 className="mx-auto h-8 w-8 animate-spin" />
                                        <p className="mt-2 text-sm text-muted-foreground">Analyzing image...</p>
                                    </div>
                                ) : (
                                    <div className="text-center">
                                        <UploadCloud className="mx-auto h-8 w-8 text-muted-foreground" />
                                        <p className="mt-2 text-sm text-muted-foreground">
                                        {isDragActive ? 'Drop the image here!' : "Drag 'n' drop an image, or click to select"}
                                        </p>
                                    </div>
                                )}
                             </div>
                           ) : (
                             <div className="relative w-full aspect-video rounded-md overflow-hidden">
                                <Image src={imagePreview} alt="Image preview" layout="fill" objectFit="cover" />
                                <Button
                                  type="button"
                                  variant="destructive"
                                  size="icon"
                                  className="absolute top-2 right-2 h-7 w-7"
                                  onClick={handleClearImage}
                                  disabled={isAnalyzing}
                                >
                                    <X className="h-4 w-4" />
                                </Button>
                             </div>
                           )}
                        </div>
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

            <FormField
                control={form.control}
                name="name"
                render={({ field }) => (
                <FormItem>
                    <FormLabel>Worker Name</FormLabel>
                    <FormControl>
                    <Input placeholder="e.g., Jane Doe" {...field} />
                    </FormControl>
                    <FormMessage />
                </FormItem>
                )}
            />
            <FormField
                control={form.control}
                name="role"
                render={({ field }) => (
                <FormItem>
                    <FormLabel>Role / Activity</FormLabel>
                    <FormControl>
                    <Input placeholder="e.g., Lead Technician" {...field} />
                    </FormControl>
                    <FormMessage />
                </FormItem>
                )}
            />

             { (imagePreview || form.getValues('description')) && (
                 <>
                    <FormField
                        control={form.control}
                        name="description"
                        render={({ field }) => (
                        <FormItem>
                            <FormLabel>AI Description</FormLabel>
                            <FormControl>
                            <Textarea placeholder="AI-generated description..." {...field} disabled={isAnalyzing} rows={3} />
                            </FormControl>
                            <FormMessage />
                        </FormItem>
                        )}
                    />
                    <FormField
                        control={form.control}
                        name="imageHint"
                        render={({ field }) => (
                        <FormItem>
                            <FormLabel>AI Image Hint</FormLabel>
                            <FormControl>
                            <Input placeholder="AI-generated hint..." {...field} disabled={isAnalyzing} />
                            </FormControl>
                            <FormMessage />
                        </FormItem>
                        )}
                    />
                 </>
             )}
            </div>
            
            <DialogFooter>
              <Button type="button" variant="outline" onClick={onClose} disabled={isSubmitting || isAnalyzing}>Cancel</Button>
              <Button type="submit" disabled={isSubmitting || isAnalyzing || !form.formState.isValid}>
                {(isSubmitting || isAnalyzing) && <Loader2 className="mr-2 h-4 w-4 animate-spin" />}
                Save Image
              </Button>
            </DialogFooter>
          </form>
        </Form>
      </DialogContent>
    </Dialog>
  );
}
