'use server';
/**
 * @fileOverview An AI flow to analyze an image and generate a description and hint.
 *
 * - analyzeImage - A function that takes an image and returns a description and hint.
 * - AnalyzeImageInput - The input type for the analyzeImage function.
 * - AnalyzeImageOutput - The return type for the analyzeImage function.
 */

import { ai } from '@/ai/genkit';
import { z } from 'genkit';

const AnalyzeImageInputSchema = z.object({
  photoDataUri: z.string().describe(
    "A photo as a data URI that must include a MIME type and use Base64 encoding. Expected format: 'data:<mimetype>;base64,<encoded_data>'."
  ),
});
export type AnalyzeImageInput = z.infer<typeof AnalyzeImageInputSchema>;

const AnalyzeImageOutputSchema = z.object({
  description: z.string().describe('A concise, descriptive caption for the image, suitable for an "alt" tag or a gallery description. Max 15 words.'),
  imageHint: z.string().describe('Two keywords separated by a space that can be used to search for a similar image. For example: "store interior", "friendly staff", "electronic components".'),
});
export type AnalyzeImageOutput = z.infer<typeof AnalyzeImageOutputSchema>;

export async function analyzeImage(input: AnalyzeImageInput): Promise<AnalyzeImageOutput> {
  return analyzeImageFlow(input);
}

const prompt = ai.definePrompt({
  name: 'analyzeImagePrompt',
  input: { schema: AnalyzeImageInputSchema },
  output: { schema: AnalyzeImageOutputSchema },
  prompt: `You are an expert at analyzing images and creating concise, useful metadata.

Analyze the following image and generate a description and a two-word search hint for it.

The description should be a short, descriptive caption, no more than 15 words.
The image hint should be exactly two keywords, separated by a space, that would be effective for finding this image on a stock photo website.

Image: {{media url=photoDataUri}}`,
});

const analyzeImageFlow = ai.defineFlow(
  {
    name: 'analyzeImageFlow',
    inputSchema: AnalyzeImageInputSchema,
    outputSchema: AnalyzeImageOutputSchema,
  },
  async (input) => {
    const { output } = await prompt(input);
    return output!;
  }
);
