
import { initializeApp } from 'firebase/app';
import { getFirestore, collection, writeBatch, doc, getDocs } from 'firebase/firestore';
import { firebaseConfig } from '../src/firebase/config';
import { initialProducts, initialGalleryImages, siteConfig as initialSiteConfig, type Comment } from '../src/lib/data';
import dotenv from 'dotenv';

dotenv.config({ path: '.env' });

const sampleComments: Omit<Comment, 'id' | 'createdAt'>[] = [
    { authorName: 'Fatu Kamara', rating: 5, text: 'This screen is crystal clear! My phone looks brand new. Tenki ya!' },
    { authorName: 'Mohamed Sesay', rating: 4, text: 'Good quality accessory, it works just as described. The delivery was fast too.' },
    { authorName: 'Aminata Conteh', rating: 5, text: 'The best power bank I have ever used. It charges my phone so fast and lasts for days.' },
    { authorName: 'Ibrahim Jalloh', rating: 5, text: 'Dis camera fine! The picture quality is sharp, even at night. I feel much safer now.' },
    { authorName: 'Zainab Turay', rating: 4, text: 'I bought the repair kit and it has everything I needed. Very useful tools.' },
    { authorName: 'Abu Bakarr', rating: 5, text: 'Wow, this microphone is professional grade. My recordings sound so clean. Big up REGIMENZ!' },
    { authorName: 'Isata Kallon', rating: 5, text: 'The MiFi is a lifesaver! I can now work from anywhere with reliable internet. Ah gladi!' },
    { authorName: 'Santigie Kargbo', rating: 4, text: 'The screen protector was easy to apply and it feels very strong. Good product.' },
    { authorName: 'Hassan Bangura', rating: 5, text: 'I love my new smartwatch. It connects to my phone without any problem. Top-notch!' },
    { authorName: 'Adama Koroma', rating: 5, text: 'This power bank is small but mighty! A must-have for anyone in Freetown.' },
    { authorName: 'Sulaiman Bah', rating: 4, text: 'The security camera installation was straightforward. The app is also easy to use.' },
    { authorName: 'Mariama Sheriff', rating: 5, text: 'This is not a duplicate, this is the original. The quality speaks for itself.' }
];

async function seedDatabase() {
  console.log('Using Firebase config:', firebaseConfig);

  const firebaseApp = initializeApp(firebaseConfig);
  const firestore = getFirestore(firebaseApp);

  try {
    const batch = writeBatch(firestore);

    // Seed Site Config
    console.log('Seeding site configuration...');
    const siteConfigRef = doc(firestore, 'siteConfig', 'default');
    const { logo, ...siteConfigWithoutLogo } = initialSiteConfig;
    batch.set(siteConfigRef, siteConfigWithoutLogo);
    console.log('Site configuration prepared for seeding.');

    // Seed Products
    if (initialProducts.length > 0) {
      console.log('Seeding products...');
      const productsCollection = collection(firestore, 'products');
      for (const product of initialProducts) {
        // Use the image 'usage' field as the document ID for products
        const newProductRef = doc(productsCollection);
        batch.set(newProductRef, product);
      }
      console.log(`${initialProducts.length} products prepared for seeding.`);
    } else {
      console.log('No initial products to seed from local data. Checking Firestore for existing products to add comments to.');
    }


    // Seed Gallery Images
    console.log('Seeding gallery images...');
    const galleryCollection = collection(firestore, 'gallery');
    for (const image of initialGalleryImages) {
        const newImageRef = doc(galleryCollection);
        batch.set(newImageRef, image);
    }
    console.log(`${initialGalleryImages.length} gallery images prepared for seeding.`);

    // Add comments to existing products
    console.log('Fetching existing products to seed with comments...');
    const productsCollectionRef = collection(firestore, 'products');
    const productSnapshot = await getDocs(productsCollectionRef);
    let commentIndex = 0;

    if (!productSnapshot.empty) {
        productSnapshot.forEach(productDoc => {
            if (commentIndex < sampleComments.length) {
                console.log(`Adding comment to product ${productDoc.id}...`);
                const comment = sampleComments[commentIndex % sampleComments.length];
                const commentsCollectionRef = collection(firestore, 'products', productDoc.id, 'comments');
                const newCommentRef = doc(commentsCollectionRef);
                batch.set(newCommentRef, { ...comment, createdAt: new Date() });
                commentIndex++;
            }
        });
        console.log(`${commentIndex} comments prepared for seeding.`);
    } else {
        console.log('No existing products found in Firestore to add comments to.');
    }


    // Commit the batch to write all data to the database
    await batch.commit();
    console.log('Database seeded successfully!');

  } catch (error) {
    console.error('Error seeding database:', error);
  } finally {
    // Disconnect and terminate the script
    process.exit(0);
  }
}

seedDatabase();
